// $Id: CSliderArray.cpp,v 1.6 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CSliderArray.hpp"
using Exponent::GUI::Controls::CSliderArray;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CSliderArray, CControl);

//	===========================================================================
CSliderArray::CSliderArray(IControlRoot *root, const long uniqueId, const CRect &area, const long widthOfSlider, const long sliderGap, const long numberOfSliders, IActionListener *listener)
			: CControl(root, uniqueId, area, listener)
			, m_handleColour(CAlphaColour::CALPHACOLOUR_ORANGE)
			, m_handleDownColour(CAlphaColour::CALPHACOLOUR_RED)
			, m_sliderValues(NULL)
			, m_numberOfSliders(numberOfSliders)
			, m_sliders(NULL)
			, m_widthOfSlider(1)
			, m_sliderGap(1)
			, m_isDragging(false)
			, m_dragIndex(0)
			, m_lastIndex(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CSliderArray);

	// Create our arrays
	m_numberOfSliders = numberOfSliders;
	m_sliderValues    = new double[m_numberOfSliders];
	m_sliders		  = new CRect[m_numberOfSliders];

	// State varibles
	m_isDragging = false;
	m_dragIndex  = 0;
	m_lastIndex  = 0;

	// Initialise our sliders
	this->createSliders(widthOfSlider, sliderGap);
}

//	===========================================================================
CSliderArray::~CSliderArray()
{
	EXPONENT_CLASS_DESTRUCTION(CSliderArray);

	FREE_ARRAY_POINTER(m_sliders);
	FREE_ARRAY_POINTER(m_sliderValues);
}

//	===========================================================================
void CSliderArray::setValue(const double value, const long index)
{
	if (value >= 0.0 && value <= 1.0 && index >= 0 && index < m_numberOfSliders)
	{
		m_sliderValues[index] = value;
		m_sliders[index].setTop((long)((1.0 - m_sliderValues[index]) * (double)m_area.getHeight()));
		m_sliders[index].setHeight(m_area.getHeight() - m_sliders[index].getTop());
		this->update();
	}
}

//	===========================================================================
void CSliderArray::drawControl(CGraphics &graphics)
{
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// Draw the background
	this->drawPrimaryImage(graphics, m_doDefaultDrawing);

	// Now draw the sliders
	this->drawSliders(graphics);
}

//	===========================================================================
void CSliderArray::setColours(CAlphaColour frameColour, CAlphaColour backgroundColour, CAlphaColour handleColour, CAlphaColour handleDownColour)
{
	this->setDefaultControlColours(backgroundColour, frameColour);
	m_handleColour	   = handleColour;
	m_handleDownColour = handleDownColour;
}

//	===========================================================================
void CSliderArray::handleLeftButtonDown(CMouseEvent &event)
{
	m_isDragging = true;

	// Determine what is being dragged
	const CPoint mousePosition = event.getMousePosition();
	m_dragIndex				   = -1;
	const long totalWidth	   = m_widthOfSlider + m_sliderGap;
	CRect area(0, 0, m_widthOfSlider, m_area.getHeight());
	for (long i = 0; i < m_numberOfSliders; i++)
	{
		area.setLeft(i * totalWidth);
		if (area.pointIsInside(mousePosition))
		{
			m_dragIndex  = i;
		}
	}

	// If we fgound something, set its values
	if (m_dragIndex != -1)
	{
		// Set the slider rectangle
		m_sliders[m_dragIndex].setTop(event.getMousePosition().getYPosition());
		m_sliders[m_dragIndex].setHeight(m_area.getHeight() - event.getMousePosition().getYPosition());

		// Set the slider value
		m_sliderValues[m_dragIndex] = 1.0 - ((double)m_sliders[m_dragIndex].getTop()) / ((double)m_area.getHeight());

		// Store the index
		m_lastIndex = m_dragIndex;

		// Notify the listener
		if (m_actionListener)
		{
			m_actionListener->handleActionEvent(CActionEvent(this, event));
		}

		// Refresh
		this->update();	
	}

	// Lock the mouse
	m_rootControl->lockControl(this);
}

//	===========================================================================
void CSliderArray::handleLeftButtonUp(CMouseEvent &event)
{
	m_lastIndex  = 0;
	m_isDragging = false;
	m_rootControl->unlockControl();
	this->update();
}

//	===========================================================================
void CSliderArray::handleMouseMovement(CMouseEvent &event)
{
	if (m_isDragging)
	{
		// Determine what is being dragged
		const CPoint mousePosition = event.getMousePosition();
		m_dragIndex				   = -1;
		const long totalWidth	   = m_widthOfSlider + m_sliderGap;
		CRect area(0, 0, m_widthOfSlider, m_area.getHeight());
		for (long i = 0; i < m_numberOfSliders; i++)
		{
			area.setLeft(i * totalWidth);
			if (area.pointIsInside(mousePosition))
			{
				m_dragIndex  = i;
			}
		}

		// If we fgound something, set its values
		if (m_dragIndex != -1)
		{
			// Set the slider rectangle
			m_sliders[m_dragIndex].setTop(event.getMousePosition().getYPosition());
			m_sliders[m_dragIndex].setHeight(m_area.getHeight() - event.getMousePosition().getYPosition());

			// Set the slider value
			m_sliderValues[m_dragIndex] = 1.0 - ((double)m_sliders[m_dragIndex].getTop()) / (double)m_area.getHeight();

			// Store the index
			m_lastIndex = m_dragIndex;

			// Notify the listener
			if (m_actionListener)
			{
				m_actionListener->handleActionEvent(CActionEvent(this, event));
			}

			// Refresh
			this->update();	
		}
	}
}

//	===========================================================================
void CSliderArray::handleMouseLeavingArea(CMouseEvent &event)
{
	m_lastIndex  = 0;
	m_isDragging = false;
	m_rootControl->unlockControl();
	this->update();
}

//	===========================================================================
void CSliderArray::drawSliders(CGraphics &graphics)
{
	for (long i = 0; i < m_numberOfSliders; i++)
	{
		if (m_isDragging && m_dragIndex == i)
		{
			graphics.getMutableBrush()->setColour(m_handleDownColour);
			graphics.fillRectangle(m_sliders[i]);
			graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
			graphics.drawRectangle(m_sliders[i]);
		}
		else
		{
			graphics.getMutableBrush()->setColour(m_handleColour);
			graphics.fillRectangle(m_sliders[i]);
			graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_BLACK);
			graphics.drawRectangle(m_sliders[i]);
		}
	}
}

//	===========================================================================
void CSliderArray::createSliders(const long widthOfSlider, const long sliderGap)
{
	m_widthOfSlider		  = widthOfSlider; 
	m_sliderGap			  = sliderGap;
	const long totalWidth = m_widthOfSlider + m_sliderGap;
	for (long i = 0; i < m_numberOfSliders; i++)
	{
		m_sliders[i].setRect((i * totalWidth), 0, m_widthOfSlider, m_area.getHeight());
		m_sliderValues[i] = 1.0;
	}
}